import { useState, useEffect, useRef } from 'react'
import { useNavigate } from 'react-router-dom'
import { Calendar, Clock, CheckCircle, MessageSquare, Star, Package, Tent, Bed, Backpack, Mountain, Trees, MapPin, Users, Award } from 'lucide-react'
import ImageSlider from '../components/ImageSlider'
import './MyRentals.css'

const MyRentals = () => {
  const navigate = useNavigate()
  const [rentals, setRentals] = useState([])

  useEffect(() => {
    const saved = localStorage.getItem('campingRentals')
    const defaultRentals = [
      {
        id: 1,
        item: 'Палатка 4-местная',
        category: 'tents',
        startDate: '2025-05-15',
        returnDate: '2025-05-20',
        status: 'active',
        review: null
      },
      {
        id: 2,
        item: 'Спальник -10°C',
        category: 'sleeping',
        startDate: '2025-04-20',
        returnDate: '2025-04-25',
        status: 'completed',
        review: {
          rating: 5,
          text: 'Отличное качество! Очень теплый и удобный спальник.'
        }
      },
      {
        id: 3,
        item: 'Рюкзак 60л',
        category: 'backpacks',
        startDate: '2025-03-10',
        returnDate: '2025-03-15',
        status: 'completed',
        review: null
      }
    ]
    
    if (saved) {
      try {
        const parsed = JSON.parse(saved)
        // Проверяем, что данные валидны и имеют даты
        const validRentals = parsed.filter(rental => 
          rental.startDate && rental.returnDate && 
          rental.startDate.match(/^\d{4}-\d{2}-\d{2}$/) &&
          rental.returnDate.match(/^\d{4}-\d{2}-\d{2}$/)
        )
        if (validRentals.length > 0) {
          setRentals(parsed)
        } else {
          // Если данные невалидны, используем дефолтные
          setRentals(defaultRentals)
          localStorage.setItem('campingRentals', JSON.stringify(defaultRentals))
        }
      } catch (e) {
        setRentals(defaultRentals)
        localStorage.setItem('campingRentals', JSON.stringify(defaultRentals))
      }
    } else {
      setRentals(defaultRentals)
      localStorage.setItem('campingRentals', JSON.stringify(defaultRentals))
    }
  }, [])

  const handleReviewSubmit = (rentalId, rating, text) => {
    const updated = rentals.map(rental => {
      if (rental.id === rentalId) {
        return {
          ...rental,
          review: { rating, text }
        }
      }
      return rental
    })
    setRentals(updated)
    localStorage.setItem('campingRentals', JSON.stringify(updated))
  }

  const handleComplete = (rentalId) => {
    const updated = rentals.map(rental => {
      if (rental.id === rentalId) {
        return {
          ...rental,
          status: 'completed'
        }
      }
      return rental
    })
    setRentals(updated)
    localStorage.setItem('campingRentals', JSON.stringify(updated))
  }

  const formatDate = (dateString) => {
    if (!dateString) return 'Не указано'
    try {
      // Правильно парсим дату в формате YYYY-MM-DD
      const [year, month, day] = dateString.split('-').map(Number)
      if (!year || !month || !day) return 'Не указано'
      
      const date = new Date(year, month - 1, day)
      if (isNaN(date.getTime())) return 'Не указано'
      
      return date.toLocaleDateString('ru-RU', { 
        day: 'numeric', 
        month: 'long', 
        year: 'numeric' 
      })
    } catch (e) {
      return 'Не указано'
    }
  }

  const getCategoryIcon = (category) => {
    switch(category) {
      case 'tents':
        return Tent
      case 'sleeping':
        return Bed
      case 'backpacks':
        return Backpack
      default:
        return Package
    }
  }

  return (
    <div className="my-rentals-page-nature">
      <div className="container">
        <section className="page-header-nature">
          <div className="header-badge-nature">
            <span>ВАШИ АРЕНДЫ</span>
          </div>
          <h1 className="page-title-nature">
            Мои <span className="title-accent-nature">аренды</span>
          </h1>
          <p className="page-description-nature">
            Управляйте арендованным оборудованием, отслеживайте даты возврата и делитесь впечатлениями о походах
          </p>
        </section>

        <ImageSlider />

        <CampingTipsSection />

        {rentals.length === 0 ? (
          <div className="empty-state-nature">
            <div className="empty-icon-wrapper-nature">
              <Package size={64} />
            </div>
            <h2 className="empty-title-nature">Нет аренд</h2>
            <p className="empty-text-nature">Создайте первую аренду оборудования</p>
            <button 
              className="empty-button-nature" 
              onClick={() => navigate('/rent')}
            >
              Арендовать оборудование
            </button>
          </div>
        ) : (
          <section className="rentals-list-nature">
            <div className="rentals-grid-nature">
              {rentals.map((rental) => (
                <RentalCard
                  key={rental.id}
                  rental={rental}
                  onReviewSubmit={handleReviewSubmit}
                  onComplete={handleComplete}
                  formatDate={formatDate}
                  getCategoryIcon={getCategoryIcon}
                />
              ))}
            </div>
          </section>
        )}
      </div>
    </div>
  )
}

const RentalCard = ({ rental, onReviewSubmit, onComplete, formatDate, getCategoryIcon }) => {
  const [rating, setRating] = useState(rental.review?.rating || 0)
  const [reviewText, setReviewText] = useState('')
  const [isSubmitting, setIsSubmitting] = useState(false)

  const handleSubmit = () => {
    if (rating === 0 || !reviewText.trim()) {
      alert('Пожалуйста, выберите рейтинг и напишите отзыв')
      return
    }
    setIsSubmitting(true)
    onReviewSubmit(rental.id, rating, reviewText)
    setIsSubmitting(false)
    setReviewText('')
    setRating(0)
  }

  const CategoryIcon = getCategoryIcon(rental.category)

  return (
    <div className={`rental-item-nature ${rental.status}`}>
      <div className="rental-header-nature">
        <div className="rental-icon-wrapper-nature">
          <CategoryIcon size={32} />
        </div>
        <div className="rental-info-nature">
          <div className="rental-number-nature">#{rental.id}</div>
          <div className={`status-badge-nature ${rental.status}`}>
            {rental.status === 'active' ? (
              <>
                <Clock size={16} />
                <span>Активна</span>
              </>
            ) : (
              <>
                <CheckCircle size={16} />
                <span>Завершена</span>
              </>
            )}
          </div>
        </div>
      </div>
      
      <h3 className="rental-title-nature">{rental.item}</h3>
      
      <div className="rental-dates-nature">
        <div className="date-item-nature">
          <Calendar size={18} />
          <div>
            <span className="date-label-nature">Начало:</span>
            <span className="date-value-nature">{formatDate(rental.startDate)}</span>
          </div>
        </div>
        <div className="date-item-nature">
          <Calendar size={18} />
          <div>
            <span className="date-label-nature">Возврат:</span>
            <span className="date-value-nature">{formatDate(rental.returnDate)}</span>
          </div>
        </div>
      </div>
      
      {rental.status === 'active' && (
        <div className="complete-section-nature">
          <button
            className="complete-btn-nature"
            onClick={() => onComplete(rental.id)}
          >
            <CheckCircle size={20} />
            <span>Завершить аренду</span>
          </button>
        </div>
      )}
      
      {rental.status === 'completed' && (
        <div className="review-section-nature">
          {rental.review ? (
            <div className="review-display-nature">
              <div className="review-stars-nature">
                {Array.from({ length: 5 }).map((_, i) => (
                  <Star
                    key={i}
                    size={20}
                    fill={i < rental.review.rating ? '#F4A460' : 'none'}
                    color={i < rental.review.rating ? '#F4A460' : '#D4C4A8'}
                  />
                ))}
              </div>
              <p className="review-text-nature">{rental.review.text}</p>
            </div>
          ) : (
            <div className="review-form-nature">
              <div className="review-header-nature">
                <MessageSquare size={20} />
                <span>Оставить отзыв о качестве оборудования</span>
              </div>
              <p className="review-hint-nature">Оцените оборудование по 5-балльной шкале и поделитесь своими впечатлениями</p>
              <div className="rating-nature">
                <span className="rating-label-nature">Рейтинг:</span>
                {Array.from({ length: 5 }).map((_, i) => (
                  <button
                    key={i}
                    type="button"
                    className="star-btn-nature"
                    onClick={() => setRating(i + 1)}
                    title={`Оценить на ${i + 1} ${i === 0 ? 'звезду' : i < 4 ? 'звезды' : 'звезд'}`}
                  >
                    <Star
                      size={28}
                      fill={i < rating ? '#F4A460' : 'none'}
                      color={i < rating ? '#F4A460' : '#D4C4A8'}
                    />
                  </button>
                ))}
                {rating > 0 && <span className="rating-value-nature">{rating} из 5</span>}
              </div>
              <div className="textarea-wrapper-nature">
                <label htmlFor={`review-text-${rental.id}`} className="textarea-label-nature">
                  Ваш отзыв:
                </label>
                <textarea
                  id={`review-text-${rental.id}`}
                  className="review-textarea-nature"
                  placeholder="Расскажите о вашем опыте использования оборудования..."
                  rows="4"
                  value={reviewText}
                  onChange={(e) => setReviewText(e.target.value)}
                />
              </div>
              <button
                className="submit-review-nature"
                onClick={handleSubmit}
                disabled={isSubmitting || rating === 0 || !reviewText.trim()}
              >
                {isSubmitting ? 'Отправка...' : 'Отправить отзыв'}
              </button>
            </div>
          )}
        </div>
      )}
    </div>
  )
}

const CampingTipsSection = () => {
  const sectionRef = useRef(null)
  const [isVisible, setIsVisible] = useState(false)

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsVisible(true)
        }
      },
      { threshold: 0.1 }
    )

    if (sectionRef.current) {
      observer.observe(sectionRef.current)
    }

    return () => {
      if (sectionRef.current) {
        observer.unobserve(sectionRef.current)
      }
    }
  }, [])

  const tips = [
    {
      icon: Mountain,
      title: 'Выбор места',
      text: 'Выбирайте ровную площадку без камней и корней. Учитывайте направление ветра и близость к воде.'
    },
    {
      icon: Trees,
      title: 'Безопасность',
      text: 'Храните еду в герметичных контейнерах, не оставляйте мусор. Соблюдайте правила разведения костра.'
    },
    {
      icon: MapPin,
      title: 'Навигация',
      text: 'Всегда имейте карту и компас. Отмечайте свой маршрут и сообщайте о планах близким.'
    },
    {
      icon: Users,
      title: 'Команда',
      text: 'Распределяйте обязанности в группе. Каждый должен знать, где находится снаряжение.'
    }
  ]

  return (
    <section 
      ref={sectionRef}
      className={`camping-tips-nature ${isVisible ? 'visible' : ''}`}
    >
      <div className="tips-header-nature">
        <div className="tips-badge-nature">
          <Award size={20} />
          <span>СОВЕТЫ БЫВАЛЫХ</span>
        </div>
        <h2 className="tips-title-nature">Секреты успешного кемпинга</h2>
        <p className="tips-description-nature">
          Полезные рекомендации для комфортного и безопасного отдыха на природе
        </p>
      </div>
      <div className="tips-grid-nature">
        {tips.map((tip, index) => {
          const Icon = tip.icon
          return (
            <div 
              key={index} 
              className="tip-card-nature"
              style={{ animationDelay: `${index * 0.1}s` }}
            >
              <div className="tip-icon-wrapper-nature">
                <Icon size={32} />
              </div>
              <h3 className="tip-title-nature">{tip.title}</h3>
              <p className="tip-text-nature">{tip.text}</p>
            </div>
          )
        })}
      </div>
    </section>
  )
}

export default MyRentals

